#!/usr/bin/env python
"""
This file is only stack render engine.
Usage:
1. Decode from instucted code
    ./dfont.py -d INST_CODE.txt > PSEUDO_CODE.txt
2. Encode from pseudo_code
    ./dfont.py -e PSEUDO_CODE.txt > INST_CODE.txt


Implement:
1 Create FONTNAME.py contain code as followed:
    1.1 import dfont, sys, os
    1.2 cvt_dict = {"NAME": VALUE, ... }
    1.3 pseudo_code_dict = {"CHARACTER_NAME": "PSEUDO CODE TRUETYPE INSTRUCTION", ...}
    1.4 if __name__ == "__main__":
            if sys.argv[1] == "--all":
                #$0 --all = ENCODE ALL CHARS
                sfd_file = os.path.basename(sys.argv[0]).split('.',1)[0] + '.sfd'
                print "Encoding all character in %s" % (sfd_file)
                dfont.inst_encode_all( sfd_file, pseudo_code_dict, cvt_dict )
            else:
                #$0 uni0E01 > x.txt = PRINT INSTRUCTION CODE OF uni0E01 TO x.txt
                print '\n'.join(dfont.inst_encode( pseudo_code_dict[sys.argv[1]], cvt_dict ))

2 Encode:
    2.1 Process all character, this script will modify FONTNAME.sfd
        Usage: ./FONTNAME.py --all
    2.2 Encode each character, for example KO_KAI="uni0E01"
        Usage: ./FONTNAME.py uni0E01 > INSTRUCTED.TXT
"""

import sys
import os

# put cvt_dict and pseudo_code_dict varirable in FONTNAME.py
# cvt value example:
#cvt_dict = {
#    "base":             10,     #0
#    "hstem":            8,      #184
#    "vstem":            96,     #154 
#    "vstem_small":      27,     #135 
#    "headstem_small":   11,     #113    =kho
#    "headdia_small":    180,    #377    =kho
#    "headhole_small":   254,    #150    =kho
#    "headneck_kho":     39,     #180    =kho
#    "hwidth_ko":        164,    #987    =ko,tho 
#    "hwidth_kho":       14,     #690    =ko,tho 
#    "hwidth_sho":       108,    #772    =sho
#    "vheight":          49,     #1208 
#    "vheight_hi":       54,     #1208   =ko 
#    "vheight_hi_kho":   54,     #1229   =kho 
#    "vheight_lo":       88,     #20 
#    "vheight_lo_sho":   256,    #6  *** NEWVALUE
#    "head_diff":        131,    #254    =head of character: bo,po
#    "beak_diff":        9,      #102    =beak of ko,tho
#    "front_space_ko":   65,     #201    =front spacing of ko,tho
#    "front_space_kho":  38,     #340    =front spacing of kho
#}

# pseudo_code example:
#pseudo_code_dict = {
#    #ko_kai
#    "uni0E01": """
#srp0    26
#mirp[rp0,min,rnd,grey] front_kai 14
#mirp[min,rnd,grey]  hstem  12
#mirp[rp0,rnd,grey]  hwidth_14 25
#mirp[min,rnd,grey]  hstem 1
#mdrp[min,rnd,grey]  27
#srp0   14
#mdrp[rp0,rnd,grey]  18
#mdrp[min,rnd,grey]  8
#mdrp[rp0,rnd,grey]  17
#mdrp[min,rnd,grey]  10
#iup[x]
#svtca[y-axis]
#mdap[rnd]  1
#ALIGNRP    13
#mirp[rp0,rnd,grey]  vheight_shoot    22
#mirp[min,rnd,grey]  vstem_curve  5
#srp1    5
#srp2    13
#sloop   10
#ip  19 8 18 9 17 10 2 25 15 12
#iup[y]
#""",
#}


#BEGIN
# inst_dict = { "COMMAND" : ("Description",diff,pops,push), ... }
# diff: 0=NOOP, 1=1BYTE, 2=2BYTE, ...
#       -1=FIRST BYTE IS ONE BYTE COUNTER,
#       -2=FIRST BYTE IS TWO BYTE COUNTER, 
#       -3=CLEAR STACK
#       -4=REQUIRE SOME PROCESSING
inst_dict = {
    "AA" :       ("Adjust Angle", 1, 1, 0),
    "ABS" :      ("ABSolute value", 0, 1, 1),
    "ADD" :      ("ADD", 1, 2, 1),
    "ALIGNPTS" : ("ALIGN Points", 2, 2, 0),
    "ALIGNRP" :  ("ALIGN to Reference Point", 1, 1, 0),
    "AND" :      ("logical AND", 1, 2, 1),
    "CALL" :     ("CALL function", 1, 1, 0),
    "CEILING" :  ("CEILING", 0, 1, 1),
    "CINDEX" :   ("Copy the INDEXed element to the top of the stack", 1, 1, -4),
    "CLEAR" :    ("CLEAR the stack", -3, 0, 0),
    "DEBUG" :    ("DEBUG call", 1, 1, 0),
    "DELTAC1" :  ("DELTA exception C1", -2, 0, 0),
    "DELTAC2" :  ("DELTA exception C2", -2, 0, 0),
    "DELTAC3" :  ("DELTA exception C3", -2, 0, 0),
    "DELTAP1" :  ("DELTA exception P1", -2, 0, 0),
    "DELTAP2" :  ("DELTA exception P2", -2, 0, 0),
    "DELTAP3" :  ("DELTA exception P3", -2, 0, 0),
    "DEPTH" :    ("DEPTH of the stack", 0, 0, 1),
    "DIV" :      ("DIVide", 1, 2, 1),
    "DUP" :      ("DUPlicate top stack element", 0, 1, 1),
    "EIF" :      ("End IF", 0, 0, 0),
    "ELSE" :     ("ELSE clause", 0, 0, 0),
    "ENDF" :     ("END Function definition", 0, 0, 0),
    "EQ" :       ("EQual", 1, 2, 1),
    "EVEN" :     ("EVEN", 0, 1, 1),
    "FDEF" :     ("Function DEFinition", 1, 1, 0),
    "FLIPOFF" :  ("set the auto FLIP Boolean to OFF", 0, 0, 0),
    "FLIPON" :   ("set the auto FLIP Boolean to ON", 0, 0, 0),
    "FLIPPT" :   ("FLIP PoinT", 1, 1, 0),
    "FLIPRGOFF" :    ("FLIP RanGe OFF", 2, 2, 0),
    "FLIPRGON" : ("FLIP RanGe ON", 2, 2, 0),
    "FLOOR" :    ("FLOOR", 2, 2, 0),
    "GC" :       ("Get Coordinate projected onto the projection vector", 0, 1, 1),
    "GETINFO" :  ("GET INFOrmation", 0, 1, 1),
    "GFV" :      ("Get Freedom Vector", 0, 0, 2),
    "GPV" :      ("Get Projection Vector", 0, 0, 2),
    "GT" :       ("Greater Than", 1, 2, 1),
    "GTEQ" :     ("Greater Than or EQual", 1, 2, 1),
    "IDEF" :     ("Instruction DEFinition", 1, 1, 0),
    "IF" :       ("IF test", 1, 1, 0),
    "INSTCTRL" : ("INSTRuction execution ConTRoL", 2, 2, 0),
    "IP" :       ("Interpolate Point", 1, 1, 0),
    "ISECT" :    ("moves point p to the InterSECTion of two lines", 5, 5, 0),
    "IUP" :      ("Interpolate Untouched Points through the outline", 0, 0, 0),
    "JMPR" :     ("JuMP Relative", 1, 1, 0),
    "JROF" :     ("Jump Relative On False", 1, 1, 0),
    "JROT" :     ("Jump Relative On True", 3, 3, 0),
    "LOOPCALL" : ("LOOP and CALL function", 2, 2, 0),
    "LT" :       ("Less Than", 1, 2, 1),
    "LTEQ" :     ("Less Than or Equal", 1, 2, 1),
    "MAX" :      ("MAXimum of top two stack elements", 1, 2, 1),
    "MD" :       ("Measure Distance", 1, 2, 1),
    "MDAP" :     ("Move Direct Absolute Point", 1, 1, 0),
    "MDRP" :     ("Move Direct Relative Point", 1, 1, 0),
    "MIAP" :     ("Move Indirect Absolute Point", 2, 2, 0),
    "MIN" :      ("MINimum of top two stack elements", 1, 2, 1),
    "MINDEX" :   ("Move the INDEXed element to the top of the stack", 1, 1, 3),
    "MIRP" :     ("Move Indirect Relative Point", 2, 2, 0),
    "MPPEM" :    ("Measure Pixels Per EM", 0, 0, 1),
    "MPS" :      ("Measure Point Size", 0, 0, 1),
    "MSIRP" :    ("Move Stack Indirect Relative Point", 1, 1, 0),
    "MUL" :      ("MULtiply", 1, 2, 1),
    "NEG" :      ("NEGate", 0, 1, 1),
    "NEQ" :      ("Not EQual", 1, 2, 1),
    "NOT" :      ("logical NOT", 0, 1, 1),
    "NPUSHB" :   ("PUSH N Bytes", -1, 0, 0),
    "NPUSHW" :   ("PUSH N Words", -1, 0, 0),
    "NROUND" :   ("No ROUNDing of value", 0, 1, 1),
    "ODD" :      ("ODD", 0, 1, 1),
    "OR" :       ("logical OR", 1, 2, 1),
    "POP" :      ("POP top stack element", 1, 1, 0),
    "PUSHB" :    ("PUSH Bytes", -4, 0, 0),
    "PUSHW" :    ("PUSH Words", -4, 0, 0),
    "RCVT" :     ("Read Control Value Table entry", 0, 1, 1),
    "RTDG" :     ("Round Down To Grid", 0, 0, 0),
    "ROFF" :     ("Round OFF", 0, 0, 0),
    "ROLL" :     ("ROLL the top three stack elements", 0, 3, 3),
    "ROUND" :    ("ROUND value", 0, 1, 1),
    "RS" :       ("Read Store", 0, 1, 1),
    "RTDG" :     ("Round To Double Grid", 0, 0, 0),
    "RTG" :      ("Round To Grid", 0, 0, 0),
    "RTHG" :     ("Round To Half Grid", 0, 0, 0),
    "RUTG" :     ("Round Up To Grid", 0, 0, 0),
    "S45ROUND" : ("Super ROUND 45 degrees", 1, 1, 0),
    "SANGW" :    ("Set Angle Weight", 1, 1, 0),
    "SCANCTRL" : ("SCAN conversion ConTRoL", 1, 1, 0),
    "SCANTYPE" : ("SCANTYPE", 1, 1, 0),
    "SCFS" : ("Sets Coordinate From the Stack using projection vector and freedom vector", 2, 2, 0),
    "SCVTCI" :   ("Set Control Value Table Cut-In", 1, 1, 0),
    "SDB" :      ("Set Delta Base in the graphics state", 1, 1, 0),
    "SDPVTL" :   ("Set Dual Projection Vector To Line", 2, 2, 0),
    "SDS" :      ("Set Delta Shift in the graphics state", 1, 1, 0),
    "SFVFS" :    ("Set Freedom Vector From Stack", 2, 2, 0),
    "SFVTCA" :   ("Set Freedom Vector To Coordinate Axis", 0, 0, 0),
    "SFVTL" :    ("Set Freedom Vector To Line", 2, 2, 0),
    "SFVTP" :    ("Set Freedom Vector To Projection Vector", 0, 0, 0),
    "SHC" :      ("SHift Contour using reference point", 1, 1, 0),
    "SHP" :      ("SHift Point using reference point", 1, 1, 0),
    "SHPIX" :    ("SHift point by a PIXel amount", 2, 2, 0),
    "SHZ" :      ("SHift Zone using reference point", 1, 1, 0),
    "SLOOP" :    ("Set LOOP variable", 1, 1, 0),
    "SMD" :      ("Set Minimum Distance", 1, 1, 0),
    "SPVFS" :    ("Set Projection Vector From Stack", 2, 2, 0),
    "SPVTCA" :   ("Set Projection Vector To Coordinate Axis", 0, 0, 0),
    "SPVTL" :    ("Set Projection Vector To Line", 2, 2, 0),
    "SROUND" :   ("Super ROUND", 1, 1, 0),
    "SRP0" :     ("Set Reference Point 0", 1, 1, 0),
    "SRP1" :     ("Set Reference Point 1", 1, 1, 0),
    "SRP2" :     ("Set Reference Point 2", 1, 1, 0),
    "SSW" :      ("Set Single Width", 1, 1, 0),
    "SSWCI" :    ("Set Single Width Cut-In", 1, 1, 0),
    "SUB" :      ("SUBtract", 1, 2, 1),
    "SVTCA" :    ("Set freedom and projection Vectors To Coordinate Axis", 0, 0, 0),
    "SWAP" :     ("SWAP the top two elements on the stack", 0, 2, 2),
    "SZP0" :     ("Set Zone Pointer 0", 1, 1, 0),
    "SZP1" :     ("Set Zone Pointer 1", 1, 1, 0),
    "SZP2" :     ("Set Zone Pointer 2", 1, 1, 0),
    "SZPS" :     ("Set Zone PointerS", 1, 1, 0),
    "UTP" :      ("UnTouch Point", 1, 1, 0),
    "WCVTF" :    ("Write Control Value Table in Funits", 1, 1, 0),
    "WCVTP" :    ("Write Control Value Table in Pixel units", 2, 2, 0),
    "WS" :       ("Write Store", 2, 2, 0),
}

stack1_command = ("NPUSHB", "NPUSHW")
stack2_command = ("PUSHB", "PUSHW")

arg_1_command = ("MDAP", "MDRP")
arg_2_command = ("MIAP", "MIRP")

delta_command = ("F_DELTA",)
#step_list USE IN SPECIAL PSUEDO FUNCTION f_delta
step_list = [-8, -7, -6, -5, -4, -3, -2, -1, 1, 2, 3, 4, 5, 6, 7, 8]

normal_command = []
for i in range(6):
    normal_command.append( [ j for j in inst_dict.keys() if inst_dict[j][1]==i ] )

#           -1=FIRST BYTE IS ONE BYTE COUNTER,
#           -2=FIRST BYTE IS TWO BYTE COUNTER, 
#           -3=CLEAR STACK
#           -4=REQUIRE PROCESS
pop_command = []
pop_command.append([])
for i in range(-1, -4, -1):
    pop_command.append( [ j for j in inst_dict.keys() if inst_dict[j][1]==i ] )


def usage(prog_name):
    print """\
Usage: %s [-d CODE | [-e] PSUEDO_CODE]
""" % (prog_name)
    return

def line_format(cmd,stack,desc):
    #STRING, LIST, STRING
    return "%-24s %-10s ;%10s" % (cmd, "  ".join(stack), desc,)

def f_delta(first_cmd, l):
    #PSUEDO COMMAND TO PROCESS DELTA FUNCTION
    #
    #first_cmd = PSUEDO COMMAND, ex: f_deltap1 = deltap1
    #l = LIST OF ARGUMENT (INCLUDE SELF UN-UPPERCASE COMMAND IN FIRST ARGUMENT)
    #
    #usage: 
    #   f_deltap 0 16+8     #deltap1 MOVE POINT 0 AT 16 PPEM +8
    #real code will be:
    #   DELTAP1  1 0 127
    #
    d1_list = []
    d2_list = []
    d3_list = []
    first_cmd = first_cmd[2:]   #TRIM F_ OUT
    l.pop(0)
    while len(l) > 0:
        point = l.pop(0)
        pstep = l.pop(0)
        if '+' in pstep or '-' in pstep:
            if '+' in pstep:
                ppem, no_of_step = pstep.split('+')
                ppem = int(ppem)
                no_of_step = int(no_of_step)
            else:
                ppem, no_of_step = pstep.split('-')
                ppem = int(ppem)
                no_of_step = -int(no_of_step)
            if ppem < 9:
                raise ValueError('ppem must greater than 9, %s: %s %s' % (first_cmd, point, pstep,))
            if abs(no_of_step) > 8:
                raise ValueError('no_of_step must be between +8 and -8, %s: %s %s' % (first_cmd, point, pstep,))
            if no_of_step == 0:
                raise ValueError('no_of_step must not be 0, %s: %s %s' % (first_cmd, point, pstep,))
            num = (ppem-9)*16 + step_list.index(no_of_step)
            if num < 256:
                d1_list.extend([point,str(num)])
            elif num < 512:
                d2_list.extend([point,str(num)])
            elif num < 1024:
                d3_list.extend([point,str(num)])
            else:
                raise ValueError('Number of delta point exceed 1024, %s' % (num,))
        else:
            raise SyntaxError('Please use + or - in f_delta second argument, %s %s' % (point, pstep,))
    new_l = []
    if d1_list:
        new_l.append(first_cmd+'1')
        new_l.append(str(len(d1_list)/2))
        new_l.extend(d1_list)
    if d2_list:
        new_l.append(first_cmd+'2')
        new_l.append(str(len(d2_list)/2))
        new_l.extend(d2_list)
    if d3_list:
        new_l.append(first_cmd+'3')
        new_l.append(str(len(d3_list)/2))
        new_l.extend(d3_list)
    return  new_l[0], new_l

def inst_decode(txt):
    """Decode TrueType Instruction code into simpler code"""
    txt_list = txt.split("\n")
    new_list = []
    commentlist = []
    stack_list = []
    sloop = 0
    i = 0
    while i < len(txt_list):
        if "[" in txt_list[i]:
            c1,c2 = txt_list[i].strip().split("[",1)
        else:
            c1,c2 = txt_list[i].strip(), ""
        #SKIP EMPTY LINE
        if c1 == "":
            i += 1
            continue
        #STACK: NPUSHB, NPUSHW
        if c1 in stack1_command:
            i += 1
            n = int(txt_list[i].strip())
            i += 1
            while n > 0:
                stack_list.append(txt_list[i].strip())
                i += 1
                n -= 1
            continue
        #STACK2: PUSHB, PUSHW
        if c1[:5] in stack2_command:
            n = int(txt_list[i].strip()[6:])
            i += 1
            while n > 0:
                stack_list.append(txt_list[i].strip())
                i += 1
                n -= 1
            continue
        temp_list = []
        #POP ONE
        if c1 in pop_command[1]:
            idx = stack_list.pop()
            temp_list = [idx]
            for j in range(int(idx)):
                temp_list.append(stack_list.pop()) 
            new_list.append(line_format(txt_list[i],temp_list,inst_dict[c1][0]))
            i += 1
            continue
        #POP PAIR
        elif c1 in pop_command[2]:
            idx = stack_list.pop()
            temp_list = [idx]
            for j in range(int(idx)):
                temp_list.append(stack_list.pop()) 
                temp_list.append(stack_list.pop()) 
            new_list.append(line_format(txt_list[i],temp_list,inst_dict[c1][0]))
            i += 1
            continue
        #POP ALL (CLEAR STACK)
        elif c1 in pop_command[3]:
            stack_list = []
            new_list.append(line_format(txt_list[i],temp_list,inst_dict[c1][0]))
            i += 1
            continue

        #NORMAL COMMAND
        if c1 in normal_command[0]:
            count = 0
        elif c1 in normal_command[1]:
            count = 1 
        elif c1 in normal_command[2]:
            count = 2 
        elif c1 in normal_command[3]:
            count = 3 
        elif c1 in normal_command[4]:
            count = 4 
        elif c1 in normal_command[5]:
            count = 5 
        else:
            count = -1
            #print 'count-1:',c1
            new_list.append('count-1:%s' % (c1,))
        if count > 0:
            if sloop > 0:
                count += sloop-1
                sloop = 0
        while count > 0:
            if stack_list == []:
                #print temp_list
                new_list.extend(temp_list)
                break
            cnum = stack_list.pop()
            temp_list.append(cnum)
            if c1 == 'SLOOP':
                sloop = int(cnum)
            count -= 1
        new_list.append(line_format(txt_list[i],temp_list,inst_dict[c1][0]))
        i += 1
    return new_list
    
def stack_format(stack_list):

    def flush(new_list, cmd_list, ind, cur_ind):
        #if len(cmd_list)==0: return [], [], cur_ind
        if ind == 0:
            temp = ['PUSHB_','NPUSHB']
        else:
            temp = ['PUSHW_','NPUSHW']
        if len(cmd_list) < 8:
            new_list.append('%s%s' % (temp[0],len(cmd_list),))
        else:
            new_list.append(temp[1])
            new_list.append(' %s' % (len(cmd_list),))
        new_list.extend(cmd_list)
        return new_list, [], cur_ind

    if len(stack_list)==0: return []
    n = 0
    new_list = []
    cmd_list = []
    stack_list.reverse()
    if int(stack_list[n]) < 256:
        ind = 0     #BYTE
    else:
        ind = 1     #WORD
    while n < len(stack_list):
        if int(stack_list[n]) < 256:
            cur_ind = 0     #BYTE
        else:
            cur_ind = 1     #WORD
        if ind == cur_ind:
            cmd_list.append(' %s' % (stack_list[n],))
        else:
            new_list, cmd_list, ind = flush(new_list, cmd_list, ind, cur_ind)    #FLUSH
            cmd_list.append(' %s' % (stack_list[n],))
        n += 1
    new_list, cmd_list, ind = flush(new_list, cmd_list, ind, cur_ind)    #FLUSH
    return new_list 

def inst_encode(txt,cvt_dict={}):
    """Encode simple code into TrueType Instruction code"""
    if cvt_dict == {}:
        print "WARNING: cvt_dict is empty, please supply cvt_dict argument"
    comment_list = ['#',';']    #COMMENT CHARACTER
    flush_list = ['SVTCA']      #FLUSH STACK
    stack_list = []
    new_list = []
    cmd_list = []
    txt_list = txt.split('\n')
    for line in txt_list:
        line = line.replace("\t"," ")
        if " " in line:
            l = line.split(" ")
        else:
            l = [line] 
        l = [ j for j in l if j != "" ]
        if l == []:
            continue
        if l[0][0] in comment_list: #BYPASS COMMENT
            continue
        for f in flush_list:    #?FLUSH STACK
            if f == l[0][:len(f)]:
                new_list.extend(stack_format(stack_list))
                stack_list = []
                new_list.extend(cmd_list)
                cmd_list = []
                break
        if '[' in l[0]:         #TO UPPERCASE
            ltemp = l[0].split('[',1)
            first_cmd = ltemp[0].upper()
            first_word = first_cmd+'['+ltemp[1]
        else:
            first_word = l[0].upper()
            first_cmd = first_word
        #CHECK SPECIAL COMMAND ---------------------------------------
        # f_delta -> f_deltap1 = deltap1, ...
        for temp in delta_command:
            if temp in first_word:
                first_word, l = f_delta(first_word, l)
    
        #END CHECK SPECIAL COMMAND -----------------------------------
        cmd_list.append(first_word)     #KEEP COMMAND
        l.pop(0)                        #PROCESS REST STACK NUM
#
        temp_arg_list = []
        for element in l:
            if element[0] in comment_list:
                break 
            else:
                temp_list = element.split('[',1)
                if element in cvt_dict.keys():                  #CHECK CVT VALUE
                    stack_list.append(cvt_dict[element])
                    temp_arg_list.append(element)
                elif temp_list[0].upper() in inst_dict.keys():  #TRAP DELTA COMMAND
                    temp_list[0] = temp_list[0].upper()
                    element = '['.join(temp_list)
                    cmd_list.append(element)
                else:
                    stack_list.append(element)
                    temp_arg_list.append(element)
        #TEST NUMBER OF PARAMETER
        if first_cmd in arg_1_command:
            if len(temp_arg_list) != 1:
                raise ValueError('%s must take 1 argument, %s' % (first_word, temp_arg_list,))
        elif first_cmd in arg_2_command:
            if len(temp_arg_list) != 2:
                raise ValueError('%s must take 2 arguments, %s' % (first_word, temp_arg_list,))
        #
    if stack_list != []:
        new_list.extend(stack_format(stack_list))
    new_list.extend(cmd_list)
    return new_list


def inst_encode_all(sfd_file, pseudo_code_dict={}, cvt_dict={}):
    """Encode all characters in pseudo_code_dict, modify FONTNAME.sfd."""
    if os.path.isfile(sfd_file):
        sfd = open(sfd_file).read().split('\n')
        sfd_bak = sfd[:]
    else:
        print "File %s.sfd is not existed, script aborted." % (sfd_file,)
        sys.exit[1]
    for chr,txt in pseudo_code_dict.iteritems():
        search_line = 'StartChar: %s' % (chr,)
        if search_line in sfd:
            line = sfd.index(search_line)
            while line < len(sfd):
                if sfd[line] == "TtInstrs:":
                    temp_sfd = sfd[:line+1]
                    #print temp_sfd[-2:]
                    temp_sfd += inst_encode(txt,cvt_dict)
                    line += 1
                    while sfd[line] != "EndTTInstrs":
                        line += 1
                    temp_sfd += sfd[line:]
                    sfd = temp_sfd[:]
                    break
                elif sfd[line] in ["Back","Fore"]:
                    temp_sfd = sfd[:line] \
                               + ["TtInstrs:"] \
                               + inst_encode(txt,cvt_dict) \
                               + ["EndTTInstrs"]+sfd[line:]
                    sfd = temp_sfd[:]
                    break
                elif sfd[line] == "EndChar":
                    print "sfd file format error at %s" % (chr,)
                    break
                else:
                    line += 1
        else:
            print "Character %s not found, script aborted." % (chr,)
            sys.exit(1)
    if sfd != sfd_bak:
        os.rename(sfd_file, "%s.bak" % (sfd_file,))
        f = open(sfd_file,"w")
        f.write("\n".join(sfd))
        f.close()
        print "%s modified." % (sfd_file,)
    else:
        print "No modified."


if __name__ == "__main__":
    if sys.argv[1] == "-d":
        #DECODE
        #$0 -d x.txt > d.txt = PRINT PSEUDO_CODE FROM x.txt TO d.txt
        f = open(sys.argv[2])
        txt = f.read()
        f.close()
        print '\n'.join(inst_decode(txt))
    elif sys.argv[1] == "-e":
        #ENCODE
        #$0 -e d.txt > x.txt = PRINT INSTRUCTED CODE FROM PSEUDO_CODE d.txt TO x.txt
        f = open(sys.argv[2])
        txt = f.read()
        f.close()
        print '\n'.join(inst_encode(txt))
    else:
        #DEFAULT IS ENCODING
        #$0 -e d.txt > x.txt = PRINT INSTRUCTED CODE FROM PSEUDO_CODE d.txt TO x.txt
        f = open(sys.argv[1])
        txt = f.read()
        f.close()
        print '\n'.join(inst_encode(txt))
 
