#!/usr/bin/env python
"""
This script arrange stack data from pseudo code to truetype instruction code.
Usage:
1. Decode from instucted code
    ./dfont.py -d INST_CODE.txt > PSEUDO_CODE.txt
2. Encode from pseudo_code
    ./dfont.py -e PSEUDO_CODE.txt > INST_CODE.txt


Implement:
1 Create FONTNAME.py contain code as followed:
    1.1 import dfont, sys, os
    1.2 cvt_dict = {"NAME": VALUE, ... }
    1.3 pseudo_code_dict = {"CHARACTER_NAME": "PSEUDO CODE TRUETYPE INSTRUCTION", ...}
    1.4 if __name__ == "__main__":
            if sys.argv[1] == "--all":
                #$0 --all = ENCODE ALL CHARS
                sfd_file = os.path.basename(sys.argv[0]).split('.',1)[0] + '.sfd'
                print "Encoding all character in %s" % (sfd_file)
                dfont.inst_encode_all( sfd_file, pseudo_code_dict, cvt_dict )
            else:
                #$0 uni0E01 > x.txt = PRINT INSTRUCTION CODE OF uni0E01 TO x.txt
                print '\n'.join(dfont.inst_encode( pseudo_code_dict[sys.argv[1]], cvt_dict ))

2 Encode:
    2.1 Process all character, this script will modify FONTNAME.sfd
        Usage: ./FONTNAME.py --all
    2.2 Encode each character, for example KO_KAI="uni0E01"
        Usage: ./FONTNAME.py uni0E01 > INSTRUCTED.TXT
"""

import sys
import os

# put cvt_dict and pseudo_code_dict varirable in FONTNAME.py
# cvt value example:
#cvt_dict = {
#    "base":             10,     #0
#    "hstem":            8,      #184
#    "vstem":            96,     #154 
#    "vstem_small":      27,     #135 
#    "headstem_small":   11,     #113    =kho
#    "headdia_small":    180,    #377    =kho
#    "headhole_small":   254,    #150    =kho
#    "headneck_kho":     39,     #180    =kho
#    "hwidth_ko":        164,    #987    =ko,tho 
#    "hwidth_kho":       14,     #690    =ko,tho 
#    "hwidth_sho":       108,    #772    =sho
#    "vheight":          49,     #1208 
#    "vheight_hi":       54,     #1208   =ko 
#    "vheight_hi_kho":   54,     #1229   =kho 
#    "vheight_lo":       88,     #20 
#    "vheight_lo_sho":   256,    #6  *** NEWVALUE
#    "head_diff":        131,    #254    =head of character: bo,po
#    "beak_diff":        9,      #102    =beak of ko,tho
#    "front_space_ko":   65,     #201    =front spacing of ko,tho
#    "front_space_kho":  38,     #340    =front spacing of kho
#}

# pseudo_code example:
#pseudo_code_dict = {
#    #ko_kai
#    "uni0E01": """
#srp0    26
#mirp[rp0,min,rnd,grey] front_kai 14
#mirp[min,rnd,grey]  hstem  12
#mirp[rp0,rnd,grey]  hwidth_14 25
#mirp[min,rnd,grey]  hstem 1
#mdrp[min,rnd,grey]  27
#srp0   14
#mdrp[rp0,rnd,grey]  18
#mdrp[min,rnd,grey]  8
#mdrp[rp0,rnd,grey]  17
#mdrp[min,rnd,grey]  10
#iup[x]
#svtca[y-axis]
#mdap[rnd]  1
#ALIGNRP    13
#mirp[rp0,rnd,grey]  vheight_shoot    22
#mirp[min,rnd,grey]  vstem_curve  5
#srp1    5
#srp2    13
#sloop   10
#ip  19 8 18 9 17 10 2 25 15 12
#iup[y]
#""",
#}


#BEGIN
# inst_dict = { "COMMAND" : ("Description",pop,push), ... }
# pop,push: n = NUMBER OF BYTE TO POPS/PUSH,
# SPECIAL:  -1 = FIRST BYTE IS COUNTER FOR BYTE
#                (USE IN n_push_stack_commands=NPUSHB,NPUSHW),
#           -2 = FIRST BYTE IS COUNTER FOR PAIR DATA (COUNTER FOR 2 BYTES)
#                (USE IN n_pop_pairs_commands=DELTAC,DELTAP),
#           -3 = GET COUNTER FROM COMMAND 
#                (USE IN direct_push_stack_commands=PUSHB,PUSHW),
#           -9 = CLEAR STACK
#          -11 = DEPEND ON SLOOP; IF SLOOP=0; THEN 1
inst_dict = {
    "AA" :       ("Adjust Angle", 1, 0),
    "ABS" :      ("ABSolute value", 1, 1),
    "ADD" :      ("ADD", 2, 1),
    "ALIGNPTS" : ("ALIGN Points", 2, 0),
    "ALIGNRP" :  ("ALIGN to Reference Point", -11, 0),
    "AND" :      ("logical AND", 2, 1),
    "CALL" :     ("CALL function", 1, 0),
    "CEILING" :  ("CEILING", 1, 1),
    #"CINDEX" :   ("Copy the INDEXed element to the top of the stack", 1, 2),
    "CINDEX" :   ("Copy the INDEXed element to the top of the stack", 1, 1),
    "CLEAR" :    ("CLEAR the stack", -9, 0),
    "DEBUG" :    ("DEBUG call", 1, 0),
    "DELTAC1" :  ("DELTA exception C1", -2, 0),
    "DELTAC2" :  ("DELTA exception C2", -2, 0),
    "DELTAC3" :  ("DELTA exception C3", -2, 0),
    "DELTAP1" :  ("DELTA exception P1", -2, 0),
    "DELTAP2" :  ("DELTA exception P2", -2, 0),
    "DELTAP3" :  ("DELTA exception P3", -2, 0),
    "DEPTH" :    ("DEPTH of the stack", 0, 1),
    "DIV" :      ("DIVide", 2, 1),
    "DUP" :      ("DUPlicate top stack element", 1, 2),
    "EIF" :      ("End IF", 0, 0),
    "ELSE" :     ("ELSE clause", 0, 0),
    "ENDF" :     ("END Function definition", 0, 0),
    "EQ" :       ("EQual", 2, 1),
    "EVEN" :     ("EVEN", 1, 1),
    "FDEF" :     ("Function DEFinition", 1, 0),
    "FLIPOFF" :  ("set the auto FLIP Boolean to OFF", -11, 0),
    "FLIPON" :   ("set the auto FLIP Boolean to ON", 0, 0),
    "FLIPPT" :   ("FLIP PoinT", 1, 0),
    "FLIPRGOFF" :    ("FLIP RanGe OFF", 2, 0),
    "FLIPRGON" : ("FLIP RanGe ON", 2, 0),
    "FLOOR" :    ("FLOOR", 1, 1),
    "GC" :       ("Get Coordinate projected onto the projection vector", 1, 1),
    "GETINFO" :  ("GET INFOrmation", 1, 1),
    "GFV" :      ("Get Freedom Vector", 0, 2),
    "GPV" :      ("Get Projection Vector", 0, 2),
    "GT" :       ("Greater Than", 2, 1),
    "GTEQ" :     ("Greater Than or EQual", 2, 1),
    "IDEF" :     ("Instruction DEFinition", 1, 0),
    "IF" :       ("IF test", 1, 0),
    "INSTCTRL" : ("INSTRuction execution ConTRoL", 2, 0),
    "IP" :       ("Interpolate Point", -11, 0),
    "ISECT" :    ("moves point p to the InterSECTion of two lines", 5, 0),
    "IUP" :      ("Interpolate Untouched Points through the outline", 0, 0),
    "JMPR" :     ("JuMP Relative", 1, 0),
    "JROF" :     ("Jump Relative On False", 1, 0),
    "JROT" :     ("Jump Relative On True", 2, 0),
    "LOOPCALL" : ("LOOP and CALL function", 2, 0),
    "LT" :       ("Less Than", 2, 1),
    "LTEQ" :     ("Less Than or Equal", 1, 2, 1),
    "MAX" :      ("MAXimum of top two stack elements", 2, 1),
    "MD" :       ("Measure Distance", 2, 1),
    "MDAP" :     ("Move Direct Absolute Point", 1, 0),
    "MDRP" :     ("Move Direct Relative Point", 1, 0),
    "MIAP" :     ("Move Indirect Absolute Point", 2, 0),
    "MIN" :      ("MINimum of top two stack elements", 2, 1),
    #"MINDEX" :   ("Move the INDEXed element to the top of the stack", 1, 1),
    "MINDEX" :   ("Move the INDEXed element to the top of the stack", 1, 0),
    "MIRP" :     ("Move Indirect Relative Point", 2, 0),
    "MPPEM" :    ("Measure Pixels Per EM", 0, 1),
    "MPS" :      ("Measure Point Size", 0, 1),
    "MSIRP" :    ("Move Stack Indirect Relative Point", 2, 0),
    "MUL" :      ("MULtiply", 2, 1),
    "NEG" :      ("NEGate", 1, 1),
    "NEQ" :      ("Not EQual", 2, 1),
    "NOT" :      ("logical NOT", 1, 1),
    "NPUSHB" :   ("PUSH N Bytes", 0, -1),
    "NPUSHW" :   ("PUSH N Words", 0, -1),
    "NROUND" :   ("No ROUNDing of value", 1, 1),
    "ODD" :      ("ODD", 1, 1),
    "OR" :       ("logical OR", 2, 1),
    "POP" :      ("POP top stack element", 1, 0),
    "PUSHB" :    ("PUSH Bytes", -3, 0),
    "PUSHW" :    ("PUSH Words", -3, 0),
    "RCVT" :     ("Read Control Value Table entry", 1, 1),
    "RDTG" :     ("Round Down To Grid", 0, 0),
    "ROFF" :     ("Round OFF", 0, 0),
    "ROLL" :     ("ROLL the top three stack elements", 3, 3),
    "ROUND" :    ("ROUND value", 1, 1),
    "RS" :       ("Read Store", 1, 1),
    "RTDG" :     ("Round To Double Grid", 0, 0),
    "RTG" :      ("Round To Grid", 0, 0),
    "RTHG" :     ("Round To Half Grid", 0, 0),
    "RUTG" :     ("Round Up To Grid", 0, 0),
    "S45ROUND" : ("Super ROUND 45 degrees", 1, 0),
    "SANGW" :    ("Set Angle Weight", 1, 0),
    "SCANCTRL" : ("SCAN conversion ConTRoL", 1, 0),
    "SCANTYPE" : ("SCANTYPE", 1, 0),
    "SCFS" : ("Sets Coordinate From the Stack using projection vector and freedom vector", 2, 0),
    "SCVTCI" :   ("Set Control Value Table Cut-In", 1, 0),
    "SDB" :      ("Set Delta Base in the graphics state", 1, 0),
    "SDPVTL" :   ("Set Dual Projection Vector To Line", 2, 0),
    "SDS" :      ("Set Delta Shift in the graphics state", 1, 0),
    "SFVFS" :    ("Set Freedom Vector From Stack", 2, 0),
    "SFVTCA" :   ("Set Freedom Vector To Coordinate Axis", 0, 0),
    "SFVTL" :    ("Set Freedom Vector To Line", 2, 0),
    "SFVTPV" :   ("Set Freedom Vector To Projection Vector", 0, 0),
    "SHC" :      ("SHift Contour using reference point", 1, 0),
    "SHP" :      ("SHift Point using reference point", -11, 0),
    "SHPIX" :    ("SHift point by a PIXel amount", -11, 0),
    "SHZ" :      ("SHift Zone using reference point", 1, 0),
    "SLOOP" :    ("Set LOOP variable", 1, 0),
    "SMD" :      ("Set Minimum Distance", 1, 0),
    "SPVFS" :    ("Set Projection Vector From Stack", 2, 0),
    "SPVTCA" :   ("Set Projection Vector To Coordinate Axis", 0, 0),
    "SPVTL" :    ("Set Projection Vector To Line", 2, 0),
    "SROUND" :   ("Super ROUND", 1, 0),
    "SRP0" :     ("Set Reference Point 0", 1, 0),
    "SRP1" :     ("Set Reference Point 1", 1, 0),
    "SRP2" :     ("Set Reference Point 2", 1, 0),
    "SSW" :      ("Set Single Width", 1, 0),
    "SSWCI" :    ("Set Single Width Cut-In", 1, 0),
    "SUB" :      ("SUBtract", 2, 1),
    "SVTCA" :    ("Set freedom and projection Vectors To Coordinate Axis", 0, 0),
    "SWAP" :     ("SWAP the top two elements on the stack", 2, 2),
    "SZP0" :     ("Set Zone Pointer 0", 1, 0),
    "SZP1" :     ("Set Zone Pointer 1", 1, 0),
    "SZP2" :     ("Set Zone Pointer 2", 1, 0),
    "SZPS" :     ("Set Zone PointerS", 1, 0),
    "UTP" :      ("UnTouch Point", 1, 0),
    "WCVTF" :    ("Write Control Value Table in Funits", 1, 0),
    "WCVTP" :    ("Write Control Value Table in Pixel units", 2, 0),
    "WS" :       ("Write Store", 2, 0),
}


#-----SPECIAL COMMAND LIST VARIABLE-----

#---DIRECT PUSH STACK COMMANDS
#NPUSHB, NPUSHW
n_push_stack_commands = [ j for j in inst_dict.keys() if inst_dict[j][2] == -1 ]
#PUSHB, PUSHW
direct_push_stack_commands = ["PUSHB","PUSHW"]

#---DIRECT POP STACK COMMANDS
#FUTURE USE
n_pop_stack_commands = [ j for j in inst_dict.keys() if inst_dict[j][1] == -1 ]
#DELTAC,DELTAP
n_pop_pairs_commands = [ j for j in inst_dict.keys() if inst_dict[j][1] == -2 ]
#CLEAR
pop_all_commands = ["CLEAR"]

#---DEPTH MEASURE COMMANDS
depth_measure_commands = ["DEPTH"]

#---LOOP COUNTER COMMANDS
loop_counter_commands = ["SLOOP"]

#---FLUSH SLOOP COMMANDS
flush_sloop_commands = ["ALIGNRP", "FLIPPT", "IP", "SHP", "SHPIX"]
 

#
arg_1_command = ("MDAP", "MDRP")
arg_2_command = ("MIAP", "MIRP")

delta_command = ("F_DELTA",)
#step_list USE IN SPECIAL PSUEDO FUNCTION f_delta
step_list = [-8, -7, -6, -5, -4, -3, -2, -1, 1, 2, 3, 4, 5, 6, 7, 8]



def usage(prog_name):
    print """\
Usage: %s [-d CODE | [-e] PSUEDO_CODE]
""" % (prog_name)
    return


def p_line_format(cmd,stack,desc):
    #STRING, LIST, STRING
    return "%-15s %-10s ;%10s" % (cmd, " ".join([str(j) for j in stack]), desc,)

def p_comment_format(cmd,stack,desc):
    return "                           #--- %-9s %-9s ;%9s" % (cmd, " ".join([str(j) for j in stack]), desc,)


#--------------------SELF MODIFY COMMAND--------------------
def f_delta(first_cmd, l):
    #PSUEDO COMMAND TO PROCESS DELTA FUNCTION
    #
    #first_cmd = PSUEDO COMMAND, ex: f_deltap1 = deltap1
    #l = LIST OF ARGUMENT (INCLUDE SELF UN-UPPERCASE COMMAND IN FIRST ARGUMENT)
    #
    #usage: 
    #   f_deltap 0 16+8     #deltap1 MOVE POINT 0 AT 16 PPEM +8
    #real code will be:
    #   DELTAP1  1 0 127
    #
    d1_list = []
    d2_list = []
    d3_list = []
    first_cmd = first_cmd[2:]   #TRIM F_ OUT
    l.pop(0)
    while len(l) > 0:
        point = l.pop(0)
        pstep = l.pop(0)
        if '+' in pstep or '-' in pstep:
            if '+' in pstep:
                ppem, no_of_step = pstep.split('+')
                ppem = int(ppem)
                no_of_step = int(no_of_step)
            else:
                ppem, no_of_step = pstep.split('-')
                ppem = int(ppem)
                no_of_step = -int(no_of_step)
            if ppem < 9:
                raise ValueError('ppem must greater than 9, %s: %s %s'\
                         % (first_cmd, point, pstep,))
            if abs(no_of_step) > 8:
                raise ValueError('no_of_step must be between +8 and -8, %s: %s %s'\
                         % (first_cmd, point, pstep,))
            if no_of_step == 0:
                raise ValueError('no_of_step must not be 0, %s: %s %s'\
                         % (first_cmd, point, pstep,))
            num = (ppem-9)*16 + step_list.index(no_of_step)
            if num < 256:
                d1_list.extend([point,str(num)])
            elif num < 512:
                d2_list.extend([point,str(num)])
            elif num < 1024:
                d3_list.extend([point,str(num)])
            else:
                raise ValueError('Number of delta point exceed 1024, %s'\
                         % (num,))
        else:
            raise SyntaxError('Please use + or - in f_delta second argument, %s %s'\
                         % (point, pstep,))
    new_l = []
    if d1_list:
        new_l.append(first_cmd+'1')
        new_l.append(str(len(d1_list)/2))
        new_l.extend(d1_list)
    if d2_list:
        new_l.append(first_cmd+'2')
        new_l.append(str(len(d2_list)/2))
        new_l.extend(d2_list)
    if d3_list:
        new_l.append(first_cmd+'3')
        new_l.append(str(len(d3_list)/2))
        new_l.extend(d3_list)
    return  new_l[0], new_l

def get_short_cmd():
    #USAGE: "mirpmrg    5" = "MIRP[min,rnd,grey]    5"
    a_dict = {
        '0':"rp0",  'M':"min",  'R':"rnd",  'N':"no-rnd",
        'W':"white",'B':"black",'G':"grey",
        }
    c1 = ["MDAP","MIAP"]
    c2 = ["MDRP","MIRP"]
    a1 = ['0','']
    a2 = ['M','']
    a3 = ['R','N']
    a3_1 = ['R','']
    a4 = ['G','W','B']
    all_short_cmd = []
    for ic in c1:
        for ia3 in a3:
            all_short_cmd.append(ic+ia3)
    for ic in c2:
        for ia1 in a1:
            for ia2 in a2:
                for ia3 in a3_1:
                    for ia4 in a4:
                        all_short_cmd.append(ic+ia1+ia2+ia3+ia4)
    sc_dict = {}
    for i in all_short_cmd:
        c = i[:4]
        a = i[4:]
        al = []
        for j in a:
            al.append(a_dict[j])
        sc_dict[i] = c + '[' + ','.join(al) + ']'

    return sc_dict

short_cmd_dict = get_short_cmd()
#USAGE: "mirpmrg    5" = "MIRP[min,rnd,grey]    5"

#----------------------------------------------------------

def inst_decode(txt,cvt_dict={},func_arg_list=[],stack_list=[]):
    #func_arg_list = [[ARG_COUNT_0,RETVAL_COUNT_0],...]
    """Decode TrueType Instruction code into simpler code"""

    def chk_empty_stack(stack_list,arg_count):
        if not stack_list:
            arg_count += 1
            stack_list = [ "&_arg_%s" % (arg_count,) ]
        return stack_list, arg_count

    def get_cvt_var(num):
        return cvt_dict.keys()[cvt_dict.values().index(num)]

    #DEFAULT IS 30 FUNCTION DEFINED
    if not func_arg_list:
        func_arg_list = [[0,0]]*30

    txt_list = txt.split("\n")
    txt_list = [i.strip() for i in txt_list] 
    txt_list = [i for i in txt_list if i]
    new_list = []
    commentlist = []
    #stack_list = []
    return_stack_list = []
    arg_count = 0           #ARGUMENT COUNTER (USE IN FPGM)
    sloop = 0               #SLOOP COUNTER
    func_count = 0          #NUMBER OF FUNCTION COUNTER

    is_fpgm = False         #IS FPGM CODE?
    stack_container = []    #KEEP STACK IN IF_CLAUSE
    arg_container = []      #KEEP NUMBER OF ARG IN IF_CLAUSE
    if_state = []           #IF_CLAUSE STATE
    else_state = []         #ELSE_CLAUSE STATE
    
    i = 0
    while i < len(txt_list):
        #SPLIT COMMAND FROM ARGUMNET
        if "[" in txt_list[i]:
            c1,c2 = txt_list[i].strip().split("[",1)
        else:
            c1,c2 = txt_list[i].strip(), ""

        p_cmd = txt_list[i]             #PRINTED COMMAND
        p_arg_list = []                 #PRINTED ARGUMENTS
        try:
            p_comment = inst_dict[c1][0]    #PRINTED COMMENT
        except:
            p_comment = inst_dict[c1[:5]][0]


        #--------------DIRECT PUSH COMMAND--------------
        #FOR NPUSHB, NPUSHW (GEN. PSEUDO CODE AS COMMENT)
        if c1 in n_push_stack_commands:
            i += 1
            n = int(txt_list[i].strip())    #GET NUMBER OF BYTE
            p_arg_list.append(n)
            i += 1
            while n > 0:
                arg = txt_list[i].strip()
                stack_list.append(arg)
                p_arg_list.append(arg)
                i += 1
                n -= 1
            #GEN. PSEUDO CODE AS COMMENT
            new_list.append(p_comment_format( p_cmd, p_arg_list, p_comment ))
            continue    #loop
        #FOR PUSHB, PUSHW (GEN. PSEUDO CODE AS COMMENT)
        if c1[:5] in direct_push_stack_commands:
            n = int(txt_list[i].strip()[6:])    #GET NUMBER OF BYTE
            i += 1
            while n > 0:
                arg = txt_list[i].strip()
                stack_list.append(arg)
                p_arg_list.append(arg)
                i += 1
                n -= 1
            #GEN.PSEUDO CODE AS COMMENT
            new_list.append(p_comment_format( p_cmd, p_arg_list, p_comment ))
            continue    #loop

        #--------------DIRECT POP COMMANDS----------------------
        #FOR FUTURE USE
        if c1 in n_pop_stack_commands:
            stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
            num = stack_list.pop()
            p_arg_list.append(num)
            for j in range(int(num)):
                stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
                s = stack_list.pop()
                p_arg_list.append(s)
            new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            continue
        #FOR DELTAC, DELTAP
        if c1 in n_pop_pairs_commands:
            stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
            num = stack_list.pop()
            p_arg_list.append(num)
            for j in range(int(num)):
                stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
                s = stack_list.pop()
                p_arg_list.append(s)
                stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
                s = stack_list.pop()
                p_arg_list.append(s)
            new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            continue
        #POP ALL (CLEAR STACK)
        if c1 in pop_all_commands:
            stack_list = []
            new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            continue

        #--------------REQUIRE REAL CALCULATION COMMANDS-------
        #FOR DEPTH
        #   ****************
        #   REQUIRE CALCULATE DEPTH OF STACK TO NUMBER
        #   BECAUSE "SLOOP" NEED INTEGER VALUE TO PROCESS
        if c1 in depth_measure_commands:
            num = len(stack_list)
            stack_list.append(num)
            p_arg_list.append( "#%s" % (num,))
            #PRINT AS COMMENT
            new_list.append(p_comment_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            continue

#wait for knowledge :P
#        #FOR DUP
#        #   ****************
#        #   REQUIRE CALCULATE DUPPLICATE OF STACK TO NUMBER
#        if c1 in ["DUP"]:
#            #DO NOT STRICKLY CALCULATE IN fpgm
#            if is_fpgm:
#                stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
#            #
#            arg = stack_list[-1]
#            stack_list.append(arg)
#            p_arg_list.append( "#%s" % (arg,))
#            #PRINT AS COMMENT
#            new_list.append(p_comment_format( p_cmd, p_arg_list, p_comment ))
#            i += 1
#            continue

        #--------------LOOP COMMANDS----------------------
        #FOR SLOOP
        if c1 in loop_counter_commands:
            stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
            num = stack_list.pop()
            p_arg_list.append(num)
            sloop += int(num)
            new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            continue
    
        #--------------FLUSH SLOOP COMMANDS----------------------
        #FOR ALIGNRP, FLIPPT, IP, SHP, SHPIX 
        if c1 in flush_sloop_commands:
            if sloop == 0:
                sloop = 1
            for j in range(sloop,0,-1):
#xxx
                if c1 in ['SHPIX']:
                    stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
                    s = stack_list.pop()
                    p_arg_list.append(s)
                stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
                s = stack_list.pop()
                p_arg_list.append(s)
            new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))
            i += 1
            sloop = 0
            continue

        #--------------COPY INDEX COMMANDS-----------------
        #FOR CINDEX
        #--------------MOVE INDEX COMMANDS-----------------
        #FOR MINDEX
        #?????

        #--------------OTHER COMMANDS----------------------
        npop = inst_dict[c1][1]     #NUMBER OF BYTE/WORDS TO POP FROM STACK
        npush = inst_dict[c1][2]    #NUMBER OF BYTE/WORDS TO PUSH INTO STACK

        #DETERMINE NUMBER OF ARGUMENT TO POP IN FUNCTION CALL
        if c1 in ["CALL"]:
            #NO OF STACK TO POP
            func_no = int(stack_list[-1])
            npop += func_arg_list[func_no][0]

        #DETERMINE NUMBER OF ARGUMENT TO POP IN FUNCTION CALL
        #AND CLEAR GARBAGE STACK

        #POP
        for j in range(npop, 0, -1):
            stack_list,arg_count = chk_empty_stack(stack_list,arg_count)
            s = stack_list.pop()
            if j == npop:
                #TRY LOOKUP VARIABLE IN CVT
                if c1 in ['MIAP','MIRP','RCVT'] and cvt_dict:
                    try:
                        s = get_cvt_var(int(s))
                    except:
                        pass

                elif c1 in ["FDEF"]:
                    stack_list = [ j for j in stack_list if str(j).isdigit() ]
                    #func_count = int(stack_list[-1])
                    func_count = int(s)
                    is_fpgm = True
                    #ASSUME THAT NO OTHER VALUE IN REMAINING STACK
                    #RETURN ALL VALUE IN STACK BACK AND CLEAR STACK
                    return_stack_list = stack_list[:]
                    stack_list = []
    
            p_arg_list.append(s)

        #CHECK IF_CLAUSE AFTER POP
        #IN IF_CLAUSE BLOCK, ONLY COLLECT STACK IN FIRST IF_CLAUSE
        if c1 in ["IF"]:
            if_state.append(1)
            else_state.append(0)
            stack_container.append(stack_list)
            arg_container.append(arg_count)
        if c1 in ['ELSE']:
            else_state[-1] = 1
            old_stack = stack_list[:]
            old_arg_count = arg_count
            stack_list = stack_container.pop()
            arg_count = arg_container.pop()
            stack_container.append(old_stack) 
            arg_container.append(old_arg_count)
        if c1 in ['EIF']:
            if else_state[-1]:
                stack_list = stack_container[-1]
                arg_count = arg_container[-1]
            stack_container.pop()
            arg_container.pop()
            if_state.pop()
            else_state.pop()    

        #PUSH
        #DETERMINE NUMBER OF ARGUMENT TO PUSH IN FUNCTION CALL
        if c1 in ["CALL"]:
            #RETURN VALUE = NO OF STACK TO PUSH
            push_count = func_arg_list[func_no][1]
            npush += push_count
            for j in range(push_count):
                stack_list.append("&_f%s_retval_%s" % (func_no,j+1,))
        else:
            for j in range(npush, 0, -1):
                #APPEND STACK AS VARIABLE
                stack_list.append( "&_%s_%s" % (c1,j,) )

        new_list.append(p_line_format( p_cmd, p_arg_list, p_comment ))

        # CLEAR ARGUMENT COUNTER AND STACK AFTER END FUNCTION 
        if c1 in ["ENDF"]:
            new_list.append("#ARG_COUNT=%s:" % (arg_count,))
            arg_count = 0
            stack_count = len(stack_list)
            new_list.append("#RET_COUNT=%s: %s"\
                % (stack_count, " ".join(stack_list),))
            stack_list = []
            if return_stack_list:
                new_list.append("#REST_STACK: %s"\
                    % (" ".join(return_stack_list),))
            #CHANGE VAR IN STACK LIST TO RETURN VALUE OF THIS FUNCTION
#            for j in range(stack_count):
#                stack_list.append("&_f%s_retval_%s"\
#                    % (func_count,j+1,))
            new_list.append("#------------------")

        i += 1

    if stack_list:
        new_list.append("#STACK_REMAIN=RETURN : "\
            + " ".join(stack_list))
        new_list.append("#------------------")

    return new_list, return_stack_list


def inst_decode_all(sfd_file):
    """Generate python code contain pseudo code from sfd file"""

    #GENERATE CVT FILE IF IT DOESN'T EXIST
    header_file = 'header_'+os.path.splitext(sfd_file.replace('-','_'))[0]
    py_header_file = header_file + '.py'

    if not os.path.isfile(py_header_file):
        gen_header_code(sfd_file)

    #IMPORT VARIABLE
    exec "from %s import *" % (header_file,)

    if test_file_exist(sfd_file):
        sfd = open(sfd_file).read().split('\n')

    line_no = 0
    lsfd = len(sfd)
    search_char = 'StartChar: '
    lsc = len(search_char)
    end_char = 'EndChar'
    search_inst = 'TtInstrs:'
    lsi = len(search_inst)
    end_inst = 'EndTTInstrs'

    pseudo_code_dict = {}

    while line_no < lsfd:
        while  line_no < lsfd and search_char not in sfd[line_no]:
            line_no += 1
        if line_no < lsfd:
            char = sfd[line_no][lsc:].strip()
        while  line_no < lsfd and search_inst not in sfd[line_no]:
            line_no += 1
        line_no += 1
        inst_list = []
        while  line_no < lsfd and sfd[line_no].strip() != end_inst:
            inst_list.append(sfd[line_no])
            line_no += 1

        l, temp_list = inst_decode(
            '\n'.join(inst_list),cvt_dict,func_arg_list)
        pseudo_code = '\n'.join(l)
        pseudo_code_dict[char] = pseudo_code
        while  line_no < lsfd and sfd[line_no].strip() != end_char:
            line_no += 1

    inst_file = 'inst_'+os.path.splitext(sfd_file.replace('-','_'))[0]+'.py'
    if os.path.isfile(inst_file):
        os.rename(inst_file, "%s.bak" % (inst_file,))
        print "Create backup file: %s.bak" % (inst_file,)

    f = open(inst_file,"w")
    header = open(py_header_file).read().split('\n')
    f.write('\n'.join(header))
    f.write("\n\npseudo_code_dict = {\n")
    pseudo_code_list = [ '''    '%s': """
%s
""",
''' % (i,pseudo_code_dict[i],) for i in pseudo_code_dict.keys() ]
    pseudo_code_list.sort()

    f.write("\n".join(pseudo_code_list))
    f.write("\n}\n")
    f.close()
    print "%s generated." % (inst_file,)


def stack_format(stack_list):

    def flush(new_list, cmd_list, ind, cur_ind):
        #FLUSH cmd_list TO new_list IN FORM ['PUSHW_?','1','2',...]
        if ind == 0:
            temp = ['PUSHB_','NPUSHB']
        else:
            temp = ['PUSHW_','NPUSHW']
        if len(cmd_list) < 8:
            new_list.append('%s%s' % (temp[0],len(cmd_list),))
        else:
            new_list.append(temp[1])
            new_list.append(' %s' % (len(cmd_list),))
        new_list.extend(cmd_list)
        return new_list, [], cur_ind

    if len(stack_list)==0: return []
    n = 0
    new_list = []
    cmd_list = []
    stack_list.reverse()
    if int(stack_list[n]) < 0:
        ind = 1     #WORD FOR MINUS
    elif int(stack_list[n]) < 256:
        ind = 0     #BYTE
    else:
        ind = 1     #WORD

    while n < len(stack_list):
        if int(stack_list[n]) < 0:
            cur_ind = 1     #WORD FOR MINUS
        elif int(stack_list[n]) < 256:
            cur_ind = 0     #BYTE
        else:
            cur_ind = 1     #WORD
        if ind == cur_ind:
            cmd_list.append(' %s' % (stack_list[n],))
        else:
            new_list, cmd_list, ind = flush(new_list, cmd_list, ind, cur_ind)    #FLUSH
            cmd_list.append(' %s' % (stack_list[n],))
        n += 1

    new_list, cmd_list, ind = flush(new_list, cmd_list, ind, cur_ind)    #FLUSH
    return new_list 


def inst_encode(txt,cvt_dict={},func_arg_list=[]):
    """Encode simple code into TrueType Instruction code"""

    if cvt_dict == {}:
        print "WARNING: cvt_dict is empty, please supply cvt_dict argument"

    warn_list = ['DEPTH','CINDEX','MINDEX']
    comment_list = ['#',';']    #COMMENT CHARACTER
    #flush_list = ['SVTCA','ENDF']      #FLUSH STACK
    flush_list = ['IUP','MD']      #FLUSH STACK
    stack_list = []
    new_list = []
    cmd_list = []
    txt_list = txt.split('\n')
    for line in txt_list:
        line = line.replace("\t"," ")
        for c in comment_list:  #TRIM COMMENT OUT
            sep = " "+c
            if sep in line:
                line,temp = line.split(sep,1)
        if " " in line:
            l = line.split(" ")
        else:
            l = [line] 

        l = [ j for j in l if j ]
        if l == []:
            continue

        if l[0][0] in comment_list: #BYPASS COMMENT
            continue

        for f in flush_list:    #?FLUSH STACK
            if f == l[0][:len(f)]:
                #STRIP VARIABLE IN stack_list (form: &_VAR)
                stack_list = [x for x in stack_list if type(x)==type(1) or len(x)<3 or (len(x)>2 and x[:2] != '&_')]
                #
                new_list.extend(stack_format(stack_list))
                stack_list = []
                new_list.extend(cmd_list)
                cmd_list = []
                break

        #-----CHECK SHORT COMMAND ---------------------
        if l[0].upper() in short_cmd_dict.keys():
            l[0] = short_cmd_dict[l[0].upper()]

        #-----END CHECK SHORT COMMAND -----------------

        if '[' in l[0]:         #TO UPPERCASE
            ltemp = l[0].split('[',1)
            first_cmd = ltemp[0].upper()
            first_word = first_cmd+'['+ltemp[1]
        else:
            first_word = l[0].upper()
            first_cmd = first_word

        #-----CHECK SPECIAL COMMAND ---------------------
        # f_delta -> f_deltap1 = deltap1, ...
        for temp in delta_command:
            if temp in first_word:
                first_word, l = f_delta(first_word, l)
        #-----END CHECK SPECIAL COMMAND -----------------

        cmd_list.append(first_word)     #KEEP COMMAND
        l.pop(0)                        #PROCESS REST STACK NUM

        temp_arg_list = []
        for element in l:
            if element[0] in comment_list:
                break 
            else:
                temp_list = element.split('[',1)
                if element in cvt_dict.keys():
                    #CHECK CVT VALUE
                    stack_list.append(cvt_dict[element])
                    temp_arg_list.append(element)
                elif temp_list[0].upper() in inst_dict.keys():
                    #TRAP DELTA COMMAND
                    temp_list[0] = temp_list[0].upper()
                    element = '['.join(temp_list)
                    cmd_list.append(element)
                else:
                    stack_list.append(element)
                    temp_arg_list.append(element)

        #TEST NUMBER OF PARAMETER
        if first_cmd in arg_1_command:
            if len(temp_arg_list) != 1:
                raise ValueError('%s must take 1 argument, %s'\
                    % (first_word, temp_arg_list,))
        elif first_cmd in arg_2_command:
            if len(temp_arg_list) != 2:
                raise ValueError('%s must take 2 arguments, %s'\
                    % (first_word, temp_arg_list,))
        #
    if stack_list != []:
        new_list.extend(stack_format(stack_list))
    new_list.extend(cmd_list)
    return new_list


def test_file_exist(file):
    if not os.path.isfile(file):
        raise ValueError("File %s does not existed, script aborted." % (file,))
    return True


def get_txt_block_in_file(file,start_char,end_char,include_tag=False):
    """Get text block in file, begin with start_char, end with end_char"""

    if test_file_exist(file):
        txt_list = open(file).read().split('\n')

    start_char = start_char.strip()
    sn = len(start_char)
    si = -1
    for i in range(len(txt_list)):
        if len(txt_list[i]) >= sn and start_char == txt_list[i][:sn]:
            si = i
            break
    if si<0:
        raise ValueError("start_char(%s) not found in %s"\
            % (start_char, file,))
            
    end_char = end_char.strip()
    en = len(end_char)
    ei = -1
    for i in range(si,len(txt_list)):
        if len(txt_list[i]) >= en and end_char == txt_list[i][:en]:
            ei = i
            break
    if ei<0:
        raise ValueError("end_char(%s) not found in %s"\
            % (end_char, file,))

    if include_tag:
        return txt_list[si:ei+1]
    else:
        return txt_list[si+1:ei]

        
def get_var_count(l):
    arg_count = 0
    ret_count = 0
    for i in l:
        if '#ARG_COUNT=' in i:
            arg_count = int(i.split('#ARG_COUNT=',1)[1].split(':',1)[0])
        if '#RET_COUNT=' in i:
            ret_count = int(i.split('#RET_COUNT=',1)[1].split(':',1)[0])
    return [arg_count,ret_count]

def gen_header_code(sfd_file):
    """Generate header python code, contain cvt, fpgm, func_arg_list"""

    header_file = 'header_'+os.path.splitext(sfd_file.replace('-','_'))[0]
    py_header_file = header_file + '.py'

    #---CVT---
    #IF EXIST OLD HEADER, USE OLD CVT
    if os.path.isfile(py_header_file):
        cvt_str_list = get_txt_block_in_file(\
            file=py_header_file,
            start_char='cvt_dict = {',
            end_char='}',
            include_tag=False)
        exec "import %s as header" % (header_file,)
        cvt_dict = header.cvt_dict
        del header
    else:
        cvt_list = []
        if test_file_exist(sfd_file):
            sfd = open(sfd_file).read().split('\n')
             
        cvt_list = get_txt_block_in_file(\
            file=sfd_file,
            start_char='ShortTable: cvt',
            end_char='EndShort',
            include_tag=False)

        cvt_str_list = ["    'cvt_%s' : %s,   #%s" \
            % (i,i,cvt_list[i].strip(),) for i in range(len(cvt_list))]
        cvt_dict = {}
        for i in range(len(cvt_list)):
            cvt_dict["cvt_%s" % (i,)] = i

    #---FPGM---
    temp = get_txt_block_in_file(
            file=sfd_file,
            start_char='TtTable: fpgm',
            end_char='EndTTInstrs',
            include_tag=False)
    fpgm_list = [i for i in '\n'.join(temp).split('ENDF') if i]
    pseudo_fpgm_list = []
    func_arg_list = []
    finish_list = []
    stack_list = []
    #--- decode 0 called function
    for i in range(len(fpgm_list)):
        fpgm_list[i] += 'ENDF'
        if fpgm_list[i].count('CALL') == 0:
            l,stack_list = inst_decode(fpgm_list[i],cvt_dict,[],stack_list)
            pseudo_fpgm_list.append(l)
            func_arg_list.append(get_var_count(pseudo_fpgm_list[i]))
            finish_list.append(i)
        else:
            l,stack_list = inst_decode(fpgm_list[i],cvt_dict,[],stack_list)
            pseudo_fpgm_list.append(l)
            func_arg_list.append([0,0])

    #--- try known func_arg_list till all
    n = len(fpgm_list)
    while n > len(finish_list):
        i = 0
        while i < n:
            if i not in finish_list:
                check_finish_list = ["CALL            %s" \
                    % (x,) for x in finish_list]
                test_pass = True
                for j in range(len(pseudo_fpgm_list[i])):
                    if pseudo_fpgm_list[i][j][:4] == 'CALL':
                        #"CALL       &f_no"
                        temp_list = [x for x in pseudo_fpgm_list[i][j].split(' ') if x]
                        f_no = int(temp_list[1])
                        if f_no not in finish_list:
                            test_pass = False
                if test_pass:
                    pseudo_fpgm_list[i],temp_list = inst_decode(fpgm_list[i],cvt_dict,func_arg_list)
                    func_arg_list[i] = get_var_count(pseudo_fpgm_list[i])
                    finish_list.append(i)
            i += 1

    pseudo_fpgm_txt_list = []
    for i in pseudo_fpgm_list:
        pseudo_fpgm_txt_list.append('\n'.join(i))

    #--- WRITE FILE ---
    if os.path.isfile(py_header_file):
        os.rename(py_header_file, "%s.bak" % (py_header_file,))
    
    f = open(py_header_file,"w")
    f.write('''#!/usr/bin/env python

func_arg_list = %s

cvt_dict = {
%s
    }

pseudo_fpgm = """
%s
"""

''' % (str(func_arg_list), 
        '\n'.join(cvt_str_list),
        '\n\n'.join(pseudo_fpgm_txt_list)))
    f.close()
    print "%s generated." % (header_file,)
    return



def inst_encode_all(sfd_file, pseudo_code_dict={}, cvt_dict={}, func_arg_list=[]):
    """Encode all characters in pseudo_code_dict, modify FONTNAME.sfd."""
    if test_file_exist(sfd_file):
        sfd = open(sfd_file).read().split('\n')
        sfd_bak = sfd[:]

    for chr,txt in pseudo_code_dict.iteritems():
        search_line = 'StartChar: %s' % (chr,)
        if search_line in sfd:
            line = sfd.index(search_line)
            while line < len(sfd):
                if sfd[line] == "TtInstrs:":
                    temp_sfd = sfd[:line+1]
                    temp_sfd += inst_encode(txt,cvt_dict,func_arg_list)
                    line += 1
                    while sfd[line] != "EndTTInstrs":
                        line += 1
                    temp_sfd += sfd[line:]
                    sfd = temp_sfd[:]
                    break
                elif sfd[line] in ["Back","Fore"]:
                    temp_sfd = sfd[:line] \
                               + ["TtInstrs:"] \
                               + inst_encode(txt,cvt_dict,func_arg_list) \
                               + ["EndTTInstrs"]+sfd[line:]
                    sfd = temp_sfd[:]
                    break
                elif sfd[line] == "EndChar":
                    print "sfd file format error at %s" % (chr,)
                    break
                else:
                    line += 1
        else:
            print "Character %s not found, script aborted." % (chr,)
            sys.exit(1)
    if sfd != sfd_bak:
        os.rename(sfd_file, "%s.bak" % (sfd_file,))
        f = open(sfd_file,"w")
        f.write("\n".join(sfd))
        f.close()
        print "%s modified." % (sfd_file,)
    else:
        print "No modified."


def get_fal_agrv(argv_3):
    #get func_argv_list from sys.argv[3]
    #input string: '1,2 2,3 3,4'
    #output list : [ [1,2],[2,3],[3,4] ]
    s = [ i for i in argv_3.split(' ') if i ] 
    s1 = []
    for i in s:
        s1.append(i.split(','))
    func_arg_list = []
    for i in s1:
        func_arg_list.append( [int(i[0]),int(i[1])] )
    return func_arg_list


if __name__ == "__main__":
    print "Please use dfont as module, use ttinst.py instead."

#DEPRECATED: USE ttinst.py INSTEAD
#    if sys.argv[1] == "-d":
#        #DECODE
#        #$0 -d x.txt "func_arg_list" > d.txt
#        # = PRINT PSEUDO_CODE FROM x.txt TO d.txt
#        #ex1: $0 -d x.txt "0,1 2,3" > d.txt
#        # "0,1 2,3" means:
#        #   function 0 has 0 argument and 1 return value, ...
#        #   function 1 has 2 argument and 3 return value, ...
#        if len(sys.argv) > 3:
#            func_arg_list = get_fal_argv(sys.argv[3])
#        else:
#            func_arg_list = []
#
#        f = open(sys.argv[2])
#        txt = f.read()
#        f.close()
#        l, temp_list = inst_decode(txt,func_arg_list=func_arg_list)
#        print '\n'.join(l)
#
#    elif sys.argv[1] == "--decode-all":
#        #DECODE ALL INSTRUCTION IN sfd FILE
#        #usage: $0 --decode-all FONTNAME
#        #output: inst_FONTNAME.py CONTAIN ALL PSEUDO CODE
#        inst_decode_all(sys.argv[2]+'.sfd')
#
#    elif sys.argv[1] == "--decode-header":
#        #GEN HEADER PYTHON CODE
#        #usage: $0 --decode-header FONTNAME
#        #output: header_FONTNAME.py CONTAIN func_arg_list,cvt and fpgm
#        gen_header_code(sys.argv[2]+'.sfd')
#
#    elif sys.argv[1] == "-e":
#        #ENCODE
#        #$0 -e d.txt > x.txt
#        # = PRINT INSTRUCTED CODE FROM PSEUDO_CODE d.txt TO x.txt
#        if len(sys.argv) > 3:
#            func_arg_list = get_fal_argv(sys.argv[3])
#        else:
#            func_arg_list = []
#
#        f = open(sys.argv[2])
#        txt = f.read()
#        f.close()
#        print '\n'.join(inst_encode(txt,func_arg_list=func_arg_list))
#
#    else:
#        #DEFAULT IS ENCODING
#        #$0 -e d.txt > x.txt 
#        # = PRINT INSTRUCTED CODE FROM PSEUDO_CODE d.txt TO x.txt
#        if len(sys.argv) > 2:
#            func_arg_list = get_fal_argv(sys.argv[2])
#        else:
#            func_arg_list = []
#
#        f = open(sys.argv[1])
#        txt = f.read()
#        f.close()
#        print '\n'.join(inst_encode(txt,func_arg_list=func_arg_list))
# 
